#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <omp.h>
#include "mpi.h"

#define ACCURACY  200000000

// Metoda Wallis'a obliczania wartosci liczby Pi bez OpenMP.
void
count_pi1(double *pi)
{
    double tmp = 1.0, a_n;
    long int i, N;

    N = ACCURACY;

    for (i = 1; i <= N; i++) {
        a_n = (double)(4.0*i*i / (4.0*i*i - 1.0));
        tmp = tmp * a_n;
    }

    *pi = (double)(2.0 * tmp);
}

// Metoda Leibniza obliczania wartosci liczby Pi bez OpenMP.
void
count_pi2(double *pi)
{
    double tmp = 0.0, a_n;
    long int i, N;

    N = ACCURACY;

    for (i = 0; i < N; i++) {
        if (i % 2 == 0) {
            a_n = (double)(1.0 / (2.0*i + 1.0));
        } else {
            a_n = (double)(-1.0 / (2.0*i + 1.0));
        }

        tmp = tmp + a_n;
    }

    *pi = (double)(4.0 * tmp);
}

// Metoda Wallis'a obliczania wartosci liczby Pi z wykorzystaniem OpenMP.

double count_pi1_omp(int istart, int i_co_ile)
{
    double tmp = 1.0, a_n;
    long int i, N;

    N = ACCURACY;

    #pragma omp parallel for default(none) shared(N,i_co_ile,istart) private(a_n,i) \
        reduction(* : tmp)
    for (i = istart; i <= N; i+=i_co_ile) {
        a_n = (double)(4.0*i*i / (4.0*i*i - 1.0));
        tmp = tmp * a_n;
    }

//    *pi = (double)(2.0 * tmp);
    return   (double)(tmp);
}

// Metoda Leibniza obliczania wartosci liczby Pi z wykorzystaniem OpenMP.

double count_pi2_omp(int istart, int i_co_ile)
{
    double tmp = 0.0, a_n;
    long int i, N;

    N = ACCURACY;

    #pragma omp parallel for default(none) shared(N,i_co_ile,istart) private(a_n,i) \
        reduction(+ : tmp)
    for (i = istart; i < N; i+=i_co_ile) {
        if (i % 2 == 0) {
            a_n = (double)(1.0 / (2.0*i + 1.0));
        } else {
            a_n = (double)(-1.0 / (2.0*i + 1.0));
        }

        tmp = tmp + a_n;
    }

//    *pi = (double)(4.0 * tmp);
    return   (double)(tmp);
}

// Program glowny.
int
main(int argc, char *argv[])
{
    double p1, p2, global_p1, global_p2;
    time_t begin_t, end_t;

    printf("Bez OpenMP:\n");

    begin_t = time(NULL);
    count_pi1(&p1);
    count_pi2(&p2);
    end_t = time( NULL);

    printf("Metoda Wallis'a Pi = %f.\n" , p1);
    printf("Metoda Leibniz'a Pi = %f.\n" , p2);
    printf("Czas wykonywania obliczen: %f.\n\n", difftime(end_t, begin_t));
    printf("Z OpenMP&MPI:\n");

    int rank,size,mpisupport,p=3;

    begin_t = time(NULL);
    MPI_Init_thread(&argc,&argv,MPI_THREAD_FUNNELED,&mpisupport);
    MPI_Comm_rank(MPI_COMM_WORLD,&rank);
    MPI_Comm_size(MPI_COMM_WORLD, &size);
    printf("Size = %d\n",size);
    printf("Rank = %d\n",rank);
    p1 = count_pi1_omp(rank+1,p);
    p2 = count_pi2_omp(rank,p);

    printf("Rank %d - Metoda Wallis'a P1 = %f.\n" ,rank, p1);
    printf("Rank %d - Metoda Leibniz'a P2 = %f.\n" ,rank, p2);

    MPI_Reduce(&p1,&global_p1,1,MPI_DOUBLE,
                MPI_PROD,0,MPI_COMM_WORLD);

    MPI_Reduce(&p2,&global_p2,1,MPI_DOUBLE,
                MPI_SUM,0,MPI_COMM_WORLD);

    end_t = time(NULL);

    if(rank ==0) {

       printf("Metoda Wallis'a Pi = %f.\n" , 2*global_p1);
       printf("Metoda Leibniz'a Pi = %f.\n" , 4*global_p2);
       printf("Czas wykonywania obliczen: %f.\n\n", difftime(end_t, begin_t));
    }
    MPI_Finalize();
    return 0;
}
